/*
 * Decompiled with CFR 0.152.
 */
package net.shrimpworks.unreal.packages;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import net.shrimpworks.unreal.packages.compression.ChunkChannel;
import net.shrimpworks.unreal.packages.compression.CompressedChunk;
import net.shrimpworks.unreal.packages.entities.NameNumber;

public class PackageReader
implements Closeable {
    private static final int READ_BUFFER = 8192;
    private static final char[] HEX_ARRAY = "0123456789ABCDEF".toCharArray();
    public final ReaderStats stats = new ReaderStats();
    private final SeekableByteChannel pgkChannel;
    private final ByteBuffer buffer;
    private SeekableByteChannel channel;
    protected int version = 0;
    protected CompressedChunk[] chunks = null;
    private final boolean cacheChunks;
    private final Map<CompressedChunk, ChunkChannel> chunkCache = new HashMap<CompressedChunk, ChunkChannel>();

    public PackageReader(SeekableByteChannel pkgChannel, boolean cacheChunks) {
        this.pgkChannel = pkgChannel;
        this.channel = pkgChannel;
        this.cacheChunks = cacheChunks;
        this.buffer = ByteBuffer.allocateDirect(8192).order(ByteOrder.LITTLE_ENDIAN);
    }

    public PackageReader(Path packageFile, boolean cacheChunks) throws IOException {
        this(FileChannel.open(packageFile, StandardOpenOption.READ), cacheChunks);
    }

    public PackageReader(SeekableByteChannel pkgChannel) {
        this(pkgChannel, false);
    }

    public PackageReader(Path packageFile) throws IOException {
        this(FileChannel.open(packageFile, StandardOpenOption.READ), false);
    }

    @Override
    public void close() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.close();
        }
        this.pgkChannel.close();
    }

    public String hash(String alg) {
        try {
            MessageDigest md = MessageDigest.getInstance(alg);
            this.pgkChannel.position(0L);
            this.buffer.clear();
            while (this.pgkChannel.read(this.buffer) > 0) {
                this.buffer.flip();
                md.update(this.buffer);
                this.buffer.clear();
            }
            return PackageReader.bytesToHex(md.digest()).toLowerCase();
        }
        catch (Exception e) {
            throw new RuntimeException("Failed to generate hash for package.", e);
        }
    }

    public void setChunks(CompressedChunk[] chunks) {
        this.chunks = chunks;
        this.stats.chunkCount = chunks.length;
    }

    public long size() {
        try {
            return this.pgkChannel.size();
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not determine size of package.");
        }
    }

    public int position() {
        return this.buffer.position();
    }

    public int currentPosition() {
        try {
            if (this.channel instanceof ChunkChannel) {
                return ((ChunkChannel)this.channel).chunk.uncompressedOffset + (int)(this.channel.position() - (long)this.buffer.remaining());
            }
            return (int)(this.channel.position() - (long)this.buffer.remaining());
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not determine current file position");
        }
    }

    public void moveTo(long pos) {
        this.moveTo(pos, false);
    }

    private void moveTo(long pos, boolean nonChunked) {
        this.moveTo(pos, nonChunked, false);
    }

    private void moveTo(long pos, boolean nonChunked, boolean keepChannel) {
        if (this.channel != this.pgkChannel && nonChunked) {
            this.channel = this.pgkChannel;
        }
        AtomicLong movePos = new AtomicLong(pos);
        if (!keepChannel && !nonChunked && this.chunks != null) {
            Optional<CompressedChunk> chunk = Arrays.stream(this.chunks).filter(c -> pos >= (long)c.uncompressedOffset && pos < (long)(c.uncompressedOffset + c.uncompressedSize)).findFirst();
            chunk.ifPresent(compressedChunk -> {
                if (!(this.channel instanceof ChunkChannel) || ((ChunkChannel)this.channel).chunk != compressedChunk) {
                    this.channel = this.loadChunk((CompressedChunk)compressedChunk, this.cacheChunks);
                }
                movePos.set(pos - (long)compressedChunk.uncompressedOffset);
            });
        }
        try {
            this.channel.position(movePos.get());
            this.buffer.clear();
            this.channel.read(this.buffer);
            this.buffer.flip();
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not move to position " + pos + " within package file", e);
        }
        finally {
            ++this.stats.moveToCount;
        }
    }

    public void moveRelative(int amount) {
        try {
            this.moveTo(this.channel.position() - (long)this.buffer.remaining() + (long)amount, false, true);
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not move by " + amount + " bytes within channel", e);
        }
        finally {
            ++this.stats.moveRelativeCount;
        }
    }

    public void ensureRemaining(int minRemaining) {
        try {
            if (this.buffer.capacity() < minRemaining) {
                throw new IllegalArgumentException("Impossible to fill buffer with " + minRemaining + " bytes");
            }
            if (this.buffer.remaining() < minRemaining) {
                this.fillBuffer();
            }
        }
        finally {
            ++this.stats.ensureRemainingCount;
        }
    }

    public void fillBuffer() {
        try {
            this.buffer.compact();
            this.channel.read(this.buffer);
            this.buffer.flip();
        }
        catch (IOException e) {
            throw new IllegalStateException("Could not read from package file", e);
        }
        finally {
            ++this.stats.fillBufferCount;
        }
    }

    public byte readByte() {
        return this.buffer.get();
    }

    public short readShort() {
        return this.buffer.getShort();
    }

    public int readInt() {
        return this.buffer.getInt();
    }

    public long readLong() {
        return this.buffer.getLong();
    }

    public float readFloat() {
        return this.buffer.getFloat();
    }

    public int readBytes(byte[] dest, int offset, int length) {
        int i;
        int start = this.currentPosition();
        for (int read = 0; read < length; read += this.currentPosition() - i) {
            if (this.buffer.remaining() < length) {
                this.fillBuffer();
            }
            i = this.currentPosition();
            this.buffer.get(dest, offset + read, Math.min(this.buffer.remaining(), length - read));
        }
        return this.currentPosition() - start;
    }

    public int readIndex() {
        if (this.version == 0) {
            throw new IllegalStateException("Version is not set");
        }
        if (this.version > 178) {
            return this.readInt();
        }
        boolean negative = false;
        int num = 0;
        int len = 6;
        for (int i = 0; i < 5; ++i) {
            boolean more;
            byte one = this.buffer.get();
            if (i == 0) {
                negative = (one & 0x80) > 0;
                more = (one & 0x40) > 0;
                num = one & 0x3F;
            } else if (i == 4) {
                num |= (one & 0x80) << len;
                more = false;
            } else {
                more = (one & 0x80) > 0;
                num |= (one & 0x7F) << len;
                len += 7;
            }
            if (!more) break;
        }
        return negative ? num * -1 : num;
    }

    public NameNumber readNameIndex() {
        if (this.version == 0) {
            throw new IllegalStateException("Version is not set");
        }
        if (this.version < 343) {
            return new NameNumber(this.readIndex());
        }
        int index = this.readIndex();
        int number = this.readInt();
        return new NameNumber(index, number);
    }

    public String readString() {
        return this.readString(-1);
    }

    public String readString(int length) {
        if (this.version == 0) {
            throw new IllegalStateException("Version is not set");
        }
        String string = "";
        if (this.version < 64) {
            byte v;
            byte[] val = new byte[255];
            int len = 0;
            while ((v = this.readByte()) != 0) {
                val[len] = v;
                len = (byte)(len + 1);
            }
            if (len > 0) {
                string = new String(Arrays.copyOfRange(val, 0, len), StandardCharsets.ISO_8859_1);
            }
        } else {
            int readLen;
            int len = this.version > 117 ? this.readIndex() : (length > -1 ? Math.min(length, this.readByte() & 0xFF) : this.readByte() & 0xFF);
            Charset charset = len < 0 ? StandardCharsets.UTF_16LE : StandardCharsets.ISO_8859_1;
            int n = readLen = len < 0 ? -(len * 2) : len;
            if (readLen != 0) {
                byte[] val = new byte[readLen];
                this.ensureRemaining(readLen);
                this.buffer.get(val);
                string = new String(val, charset);
            }
        }
        return string.trim();
    }

    private static String bytesToHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int i = 0; i < bytes.length; ++i) {
            int v = bytes[i] & 0xFF;
            hexChars[i * 2] = HEX_ARRAY[v >>> 4];
            hexChars[i * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars);
    }

    public ChunkChannel loadChunk(CompressedChunk chunk) {
        return this.loadChunk(chunk, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ChunkChannel loadChunk(CompressedChunk chunk, boolean cache) {
        try {
            Supplier<ChunkChannel> chunkLoader = () -> {
                try {
                    this.moveTo(chunk.compressedOffset, true);
                    if (this.readInt() != -1641380927) {
                        throw new IllegalStateException("Chunk does not seem to be Unreal package data");
                    }
                    int blockSize = this.readInt();
                    int compressedSize = this.readInt();
                    int uncompressedSize = this.readInt();
                    int numBlocks = (uncompressedSize + blockSize - 1) / blockSize;
                    int[] blockSizes = new int[numBlocks * 2];
                    for (int i = 0; i < blockSizes.length; i += 2) {
                        blockSizes[i] = this.readInt();
                        blockSizes[i + 1] = this.readInt();
                    }
                    ChunkChannel chunkChannel = new ChunkChannel(this, chunk, uncompressedSize, blockSizes);
                    return chunkChannel;
                }
                finally {
                    ++this.stats.chunkLoadCount;
                }
            };
            ChunkChannel chunkChannel = !cache ? chunkLoader.get() : this.chunkCache.computeIfAbsent(chunk, c -> (ChunkChannel)chunkLoader.get());
            return chunkChannel;
        }
        finally {
            ++this.stats.chunkFetchCount;
        }
    }

    public static class ReaderStats {
        public int moveToCount;
        public int moveRelativeCount;
        public int ensureRemainingCount;
        public int fillBufferCount;
        public int chunkCount;
        public int chunkLoadCount;
        public int chunkFetchCount;

        public String toString() {
            return String.format("ReaderStats [moveToCount=%s, moveRelativeCount=%s, ensureRemainingCount=%s, fillBufferCount=%s, chunkCount=%s, chunkLoadCount=%s, chunkFetchCount=%s]", this.moveToCount, this.moveRelativeCount, this.ensureRemainingCount, this.fillBufferCount, this.chunkCount, this.chunkLoadCount, this.chunkFetchCount);
        }
    }
}

