/*
 * Decompiled with CFR 0.152.
 */
package org.unrealarchive.indexing;

import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.unrealarchive.common.Util;
import org.unrealarchive.common.YAML;
import org.unrealarchive.content.Download;
import org.unrealarchive.content.Games;
import org.unrealarchive.content.addons.Addon;
import org.unrealarchive.content.addons.SimpleAddonRepository;
import org.unrealarchive.content.addons.SimpleAddonType;
import org.unrealarchive.indexing.AddonClassifier;
import org.unrealarchive.indexing.ContentManager;
import org.unrealarchive.indexing.Incoming;
import org.unrealarchive.indexing.IndexLog;
import org.unrealarchive.indexing.IndexResult;
import org.unrealarchive.indexing.IndexUtils;
import org.unrealarchive.indexing.Submission;
import org.unrealarchive.indexing.SubmissionOverride;

public class Indexer {
    static final Set<String> INCLUDE_TYPES = Set.of("zip", "rar", "ace", "7z", "cab", "tgz", "gz", "tar", "bz2", "exe", "umod", "ut2mod", "ut4mod");
    private final SimpleAddonRepository repo;
    private final ContentManager contentManager;
    private final IndexerEvents events;
    private final IndexerPostProcessor postProcessor;

    public Indexer(SimpleAddonRepository repo, ContentManager contentManager, IndexerEvents events) {
        this(repo, contentManager, events, new IndexerPostProcessor(){});
    }

    public Indexer(SimpleAddonRepository repo, ContentManager contentManager, IndexerEvents events, IndexerPostProcessor postProcessor) {
        this.repo = repo;
        this.contentManager = contentManager;
        this.events = events;
        this.postProcessor = postProcessor;
    }

    public void index(boolean force, boolean newOnly, int concurrency, SimpleAddonType forceType, Games forceGame, Path ... inputPath) throws IOException {
        ArrayList indexLogs = new ArrayList();
        AtomicInteger done = new AtomicInteger();
        AtomicInteger allFound = new AtomicInteger();
        LinkedBlockingDeque all = new LinkedBlockingDeque();
        CompletableFuture<Void> filesTask = CompletableFuture.runAsync(() -> {
            for (Path p : inputPath) {
                try {
                    this.findFiles(p, newOnly, all, allFound);
                }
                catch (IOException ex) {
                    throw new RuntimeException("Failed to find files in path " + String.valueOf(p), ex);
                }
            }
        });
        CompletableFuture[] workers = new CompletableFuture[concurrency];
        for (int i = 0; i < concurrency; ++i) {
            workers[i] = CompletableFuture.runAsync(() -> {
                do {
                    try {
                        Submission sub = (Submission)all.pollFirst(500L, TimeUnit.MILLISECONDS);
                        if (sub == null) continue;
                        if (forceGame != null) {
                            sub.override.overrides.put("game", forceGame.name);
                        }
                        IndexLog log = new IndexLog();
                        indexLogs.add(log);
                        this.indexFile(sub, log, force, forceType, result -> {
                            this.events.indexed(sub, (Optional<IndexResult<? extends Addon>>)result, log);
                            this.events.progress(done.incrementAndGet(), allFound.get(), sub.filePath);
                        });
                    }
                    catch (InterruptedException e) {
                        System.err.printf("Error encountered while processing index queue: %s%n", e.getMessage());
                    }
                } while (!filesTask.isDone() || !all.isEmpty());
            });
        }
        CompletableFuture.allOf(workers).join();
        int errorCount = 0;
        for (IndexLog l : indexLogs) {
            if (l.ok()) continue;
            ++errorCount;
        }
        this.events.completed(indexLogs.size(), errorCount);
    }

    private void findFiles(Path inputPath, final boolean newOnly, final Deque<Submission> all, final AtomicInteger allFound) throws IOException {
        if (Files.isDirectory(inputPath, new LinkOption[0])) {
            Files.walkFileTree(inputPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){
                final Map<Path, SubmissionOverride> overrides = new HashMap<Path, SubmissionOverride>();

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    try {
                        if (INCLUDE_TYPES.contains(Util.extension((Path)file).toLowerCase())) {
                            Submission sub;
                            if (newOnly && Indexer.this.repo.forHash(Util.hash((Path)file)) != null) {
                                return FileVisitResult.CONTINUE;
                            }
                            Path subFile = Paths.get(String.valueOf(file) + ".yml", new String[0]);
                            if (Files.exists(subFile, new LinkOption[0])) {
                                sub = (Submission)YAML.fromFile((Path)subFile, Submission.class);
                                sub.filePath = file;
                            } else {
                                sub = new Submission(file, new String[0]);
                            }
                            SubmissionOverride override = Indexer.this.findOverride(file, this.overrides);
                            if (override != null) {
                                sub.override = override;
                            }
                            all.addLast(sub);
                            allFound.incrementAndGet();
                        }
                    }
                    catch (Throwable t) {
                        throw new IOException("Failed to read file " + String.valueOf(file), t);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    if (Files.exists(dir.resolve("_override.yml"), new LinkOption[0])) {
                        this.overrides.put(dir, (SubmissionOverride)YAML.fromFile((Path)dir.resolve("_override.yml"), SubmissionOverride.class));
                    }
                    return super.preVisitDirectory(dir, attrs);
                }
            });
        } else {
            Submission sub;
            if (newOnly && this.repo.forHash(Util.hash((Path)inputPath)) != null) {
                return;
            }
            Path subFile = Paths.get(String.valueOf(inputPath) + ".yml", new String[0]);
            if (Files.exists(subFile, new LinkOption[0])) {
                sub = (Submission)YAML.fromFile((Path)subFile, Submission.class);
                sub.filePath = inputPath;
            } else {
                sub = new Submission(inputPath, new String[0]);
            }
            SubmissionOverride override = this.findOverride(inputPath, new HashMap<Path, SubmissionOverride>());
            if (override != null) {
                sub.override = override;
            }
            all.add(sub);
            allFound.incrementAndGet();
        }
    }

    private SubmissionOverride findOverride(Path file, Map<Path, SubmissionOverride> override) {
        Path parent = file.getParent();
        if (override.containsKey(parent)) {
            return override.get(parent);
        }
        SubmissionOverride result = null;
        while (parent != null) {
            if (override.containsKey(parent)) {
                result = override.get(parent);
                break;
            }
            try {
                if (Files.exists(parent.resolve("_override.yml"), new LinkOption[0])) {
                    result = (SubmissionOverride)YAML.fromFile((Path)parent.resolve("_override.yml"), SubmissionOverride.class);
                    override.put(parent, result);
                    break;
                }
            }
            catch (IOException e) {
                throw new RuntimeException("Failed to read override file in path " + String.valueOf(parent), e);
            }
            parent = parent.getParent();
        }
        if (result != null) {
            for (parent = file.getParent(); parent != null && !override.containsKey(parent); parent = parent.getParent()) {
                override.put(parent, result);
            }
        }
        return result;
    }

    private void indexFile(Submission sub, IndexLog log, boolean force, SimpleAddonType forceType, Consumer<Optional<IndexResult<? extends Addon>>> done) {
        try (Incoming incoming = new Incoming(sub, log);){
            this.identifyContent(incoming, force, forceType, (ident, content) -> {
                if (content == null || ident.contentType() == SimpleAddonType.UNKNOWN) {
                    log.log(IndexLog.EntryType.CONTINUE, String.format("No content identified in %s", sub.filePath.getFileName()));
                    done.accept(Optional.empty());
                    return;
                }
                ident.indexer().get().index(incoming, (Addon)content, result -> {
                    try {
                        Addon existing;
                        Addon current = this.repo.forHash(incoming.hash);
                        ((Addon)result.content).name = ((Addon)result.content).name.trim();
                        ((Addon)result.content).author = ((Addon)result.content).author.trim();
                        if (current == null && (existing = (Addon)this.repo.search(((Addon)result.content).game, ((Addon)result.content).contentType, ((Addon)result.content).name, ((Addon)result.content).author).stream().max(Comparator.comparing(a -> a.releaseDate)).orElse(null)) != null) {
                            if (existing.variationOf == null && existing.releaseDate.compareTo(((Addon)result.content).releaseDate) < 0) {
                                Addon variation = this.contentManager.checkout(existing.hash);
                                variation.variationOf = ((Addon)result.content).hash;
                                this.contentManager.checkin(new IndexResult<Addon>(variation, Collections.emptySet()), null);
                                log.log(IndexLog.EntryType.CONTINUE, String.format("Flagging original content %s variation", existing.originalFilename));
                            } else {
                                ((Addon)result.content).variationOf = existing.hash;
                                log.log(IndexLog.EntryType.CONTINUE, String.format("Flagging as variation of %s", existing.originalFilename));
                            }
                        }
                        ((Addon)result.content).dependencies = IndexUtils.dependencies(result.content, incoming);
                        this.postProcessor.indexed(sub, current, (IndexResult<? extends Addon>)result);
                        if (((Addon)result.content).name.isEmpty()) {
                            throw new IllegalStateException("Name cannot be blank for " + String.valueOf(incoming.submission.filePath));
                        }
                        if (current != null) {
                            result.files.removeIf(f -> {
                                if (current.attachments.stream().anyMatch(a -> a.name.equals(f.name()))) {
                                    try {
                                        Files.deleteIfExists(f.path());
                                    }
                                    catch (IOException e) {
                                        log.log(IndexLog.EntryType.CONTINUE, "Failed to delete duplicate attachment" + String.valueOf(f), e);
                                    }
                                    return true;
                                }
                                return false;
                            });
                        }
                        this.contentManager.checkin((IndexResult<? extends Addon>)result, incoming.submission);
                    }
                    catch (IOException e) {
                        log.log(IndexLog.EntryType.FATAL, "Failed to store content file data for " + sub.filePath.toString(), e);
                    }
                    done.accept(Optional.of(result));
                });
            });
        }
        catch (Throwable e) {
            log.log(IndexLog.EntryType.FATAL, e.getMessage(), e);
            done.accept(Optional.empty());
        }
    }

    private void identifyContent(Incoming incoming, boolean force, SimpleAddonType forceType, BiConsumer<AddonClassifier.AddonIdentifier, Addon> done) throws IOException {
        AddonClassifier.AddonIdentifier ident;
        Object content = this.contentManager.checkout(incoming.hash);
        if (content != null && !force) {
            if (!content.deleted && incoming.submission.sourceUrls != null) {
                for (String url : incoming.submission.sourceUrls) {
                    if (url == null || url.isEmpty() || content.hasDownload(url)) continue;
                    content.downloads.add(new Download(url));
                }
                this.contentManager.checkin(new IndexResult<Addon>((Addon)content, Collections.emptySet()), incoming.submission);
            }
            content = null;
        }
        incoming.prepare();
        AddonClassifier.AddonIdentifier addonIdentifier = ident = forceType == null ? AddonClassifier.classify(incoming) : AddonClassifier.identifierForType(forceType);
        if (content == null || !ident.contentType().toString().equalsIgnoreCase(content.contentType)) {
            content = AddonClassifier.newContent(ident, incoming);
        }
        done.accept(ident, (Addon)content);
    }

    public static interface IndexerEvents {
        public void starting(int var1);

        public void progress(int var1, int var2, Path var3);

        public void indexed(Submission var1, Optional<IndexResult<? extends Addon>> var2, IndexLog var3);

        public void completed(int var1, int var2);
    }

    public static interface IndexerPostProcessor {
        default public void indexed(Submission sub, Addon before, IndexResult<? extends Addon> result) {
            if (sub.sourceUrls != null) {
                for (String url : sub.sourceUrls) {
                    if (url == null || url.isEmpty() || result.content.hasDownload(url)) continue;
                    ((Addon)result.content).downloads.add(new Download(url));
                }
            }
        }
    }

    public static class CLIEventPrinter
    implements IndexerEvents {
        private final boolean verbose;

        public CLIEventPrinter(boolean verbose) {
            this.verbose = verbose;
        }

        @Override
        public void starting(int foundFiles) {
            System.out.printf("Found %d file(s) to index%n", foundFiles);
        }

        @Override
        public void progress(int indexed, int total, Path currentFile) {
            System.out.printf("Completed %d of %d\r", indexed, total);
        }

        @Override
        public void indexed(Submission submission, Optional<IndexResult<? extends Addon>> indexed, IndexLog log) {
            for (IndexLog.LogEntry l : log.log) {
                System.out.printf("[%s] %s: %s%n", new Object[]{l.type, Util.fileName((Path)submission.filePath.getFileName()), l.message});
                if (l.exception == null || !this.verbose) continue;
                l.exception.printStackTrace(System.out);
            }
        }

        @Override
        public void completed(int indexedFiles, int errorCount) {
            System.out.printf("%nCompleted indexing %d files, with %d errors%n", indexedFiles, errorCount);
        }
    }
}

