/*
 * Decompiled with CFR 0.152.
 */
package org.unrealarchive.submitter;

import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.PushCommand;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.PersonIdent;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.lib.TextProgressMonitor;
import org.eclipse.jgit.transport.CredentialsProvider;
import org.eclipse.jgit.transport.UsernamePasswordCredentialsProvider;
import org.kohsuke.github.GHPullRequest;
import org.kohsuke.github.GHRepository;
import org.kohsuke.github.GitHub;
import org.kohsuke.github.GitHubBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.unrealarchive.common.ArchiveUtil;
import org.unrealarchive.common.CLI;
import org.unrealarchive.common.Util;
import org.unrealarchive.content.Games;
import org.unrealarchive.content.addons.Addon;
import org.unrealarchive.content.addons.SimpleAddonRepository;
import org.unrealarchive.content.addons.SimpleAddonType;
import org.unrealarchive.indexing.ContentManager;
import org.unrealarchive.indexing.IndexLog;
import org.unrealarchive.indexing.IndexResult;
import org.unrealarchive.indexing.Indexer;
import org.unrealarchive.indexing.Scanner;
import org.unrealarchive.indexing.Submission;
import org.unrealarchive.storage.DataStore;
import org.unrealarchive.submitter.Submissions;

public class ContentRepository
implements Closeable {
    private static final String SUBMISSION_URL = "https://unrealarchive.org/submit";
    private static final Logger logger = LoggerFactory.getLogger(ContentRepository.class);
    private static final Duration GIT_POLL_TIME = Duration.ofMinutes(30L);
    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    private static final String GIT_DEFUALT_BRANCH = "master";
    private final Path tmpDir = Files.createTempDirectory("ua-submit-data-", new FileAttribute[0]);
    private final ScheduledFuture<?> schedule;
    private final String repoUrl;
    private final Git gitRepo;
    private final CredentialsProvider gitCredentials;
    private final PersonIdent gitAuthor;
    private final GHRepository repository;
    private SimpleAddonRepository contentRepo;
    private ContentManager content;
    private volatile boolean contentLock = false;

    public ContentRepository(String gitRepoUrl, String gitAuthUsername, String gitAuthPassword, String gitUserEmail, String githubToken, ScheduledExecutorService executor) throws IOException, GitAPIException {
        logger.info("Cloning git repository {} into {}", (Object)gitRepoUrl, (Object)this.tmpDir);
        this.repoUrl = gitRepoUrl;
        this.gitCredentials = new UsernamePasswordCredentialsProvider(gitAuthUsername, gitAuthPassword);
        this.gitAuthor = new PersonIdent(gitAuthUsername, gitUserEmail);
        this.gitRepo = ((CloneCommand)Git.cloneRepository().setCredentialsProvider(this.gitCredentials)).setURI(gitRepoUrl).setBranch(GIT_DEFUALT_BRANCH).setDirectory(this.tmpDir.toFile()).setDepth(1).setCloneAllBranches(false).setProgressMonitor((ProgressMonitor)new TextProgressMonitor()).call();
        Pattern repoPattern = Pattern.compile(".*/(.*)/(.*)\\.git");
        Matcher repoNameMatch = repoPattern.matcher(gitRepoUrl);
        if (!repoNameMatch.find()) {
            throw new IllegalArgumentException(String.format("Could not find repo organisation and name in input %s", gitRepoUrl));
        }
        GitHub gitHub = new GitHubBuilder().withOAuthToken(githubToken).build();
        this.repository = gitHub.getRepository(String.format("%s/%s", repoNameMatch.group(1), repoNameMatch.group(2)));
        this.contentRepo = this.initContentRepo(this.tmpDir);
        this.content = this.initContentManager(this.contentRepo);
        this.schedule = executor.scheduleWithFixedDelay(() -> {
            if (this.contentLock) {
                return;
            }
            try {
                ObjectId old = this.gitRepo.getRepository().findRef(GIT_DEFUALT_BRANCH).getObjectId();
                this.gitRepo.pull().call();
                if (!old.equals((AnyObjectId)this.gitRepo.getRepository().findRef(GIT_DEFUALT_BRANCH).getObjectId())) {
                    this.contentRepo = this.initContentRepo(this.tmpDir);
                    this.content = this.initContentManager(this.contentRepo);
                }
            }
            catch (IOException | GitAPIException e) {
                e.printStackTrace();
            }
        }, GIT_POLL_TIME.toMillis(), GIT_POLL_TIME.toMillis(), TimeUnit.MILLISECONDS);
        logger.info("Content repo started");
    }

    @Override
    public void close() {
        this.schedule.cancel(false);
        try {
            logger.info("Cleaning data path {}", (Object)this.tmpDir);
            ArchiveUtil.cleanPath((Path)this.tmpDir);
        }
        catch (IOException e) {
            logger.error("Cleanup failed", (Throwable)e);
        }
    }

    private SimpleAddonRepository initContentRepo(Path path) throws IOException {
        return new SimpleAddonRepository.FileRepository(path.resolve("content"));
    }

    private ContentManager initContentManager(SimpleAddonRepository repo) throws IOException {
        return new ContentManager(repo, this.store(DataStore.StoreContent.CONTENT), this.store(DataStore.StoreContent.IMAGES));
    }

    private DataStore store(DataStore.StoreContent contentType) {
        String stringType = System.getenv().getOrDefault("STORE_" + contentType.name().toUpperCase(), System.getenv().getOrDefault("STORE", "NOP"));
        DataStore.StoreType storeType = DataStore.StoreType.valueOf((String)stringType.toUpperCase());
        return storeType.newStore(contentType, new CLI(EMPTY_STRING_ARRAY, Map.of(), Set.of()));
    }

    public void lock() {
        if (this.contentLock) {
            throw new IllegalStateException("Already locked");
        }
        this.contentLock = true;
    }

    public void unlock() {
        this.contentLock = false;
    }

    public Set<Scanner.ScanResult> scan(final Submissions.Job job, final Path ... paths) throws IOException {
        if (paths == null || paths.length == 0) {
            throw new IllegalArgumentException("No paths to index");
        }
        Scanner sc = new Scanner(this.contentRepo, new CLI(EMPTY_STRING_ARRAY, Map.of(), Set.of()));
        final HashSet<Scanner.ScanResult> scanResults = new HashSet<Scanner.ScanResult>();
        sc.scan(new Scanner.ScannerEvents(){

            public void starting(int foundFiles, Pattern included, Pattern excluded) {
                job.log(Submissions.JobState.SCANNING, "Begin scanning content");
                logger.info("[{}] Start scanning paths {}", (Object)job.id, (Object)Arrays.toString(paths));
            }

            public void progress(int scanned, int total, Path currentFile) {
                logger.info("[{}] Scanned {} of {}", new Object[]{job.id, scanned, total});
            }

            public void scanned(Scanner.ScanResult scanned) {
                String fName = Util.fileName((Path)scanned.filePath());
                if (scanned.failed() != null) {
                    job.log(String.format("Error scanning file %s", fName), scanned.failed());
                } else if (scanned.known()) {
                    job.log(String.format("No new content found in file %s", fName), Submissions.LogType.WARN);
                } else if (scanned.newType() == SimpleAddonType.UNKNOWN) {
                    job.log(String.format("No recognisable content found in file %s", fName), Submissions.LogType.ERROR);
                } else {
                    job.log(String.format("Found a %s in file %s", scanned.newType(), fName));
                    scanResults.add(scanned);
                }
            }

            public void completed(int scannedFiles) {
                logger.info("[{}] Completed scanning {}", (Object)job.id, (Object)scannedFiles);
            }
        }, paths);
        if (scanResults.isEmpty()) {
            job.log(Submissions.JobState.SCAN_FAILED, "No new content found", Submissions.LogType.ERROR);
        } else {
            job.log(Submissions.JobState.SCANNED, "Scan completed");
        }
        return scanResults;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<IndexResult<? extends Addon>> submit(Submissions.Job job, Path ... paths) throws GitAPIException {
        if (paths == null || paths.length == 0) {
            throw new IllegalArgumentException("No paths to index");
        }
        String branchName = String.format("%s_%s", Util.slug((String)paths[0].getFileName().toString()), job.id);
        try {
            job.log(String.format("Checkout content data branch %s", branchName));
            this.checkout(branchName, true);
            HashSet<IndexResult<? extends Addon>> indexResults = new HashSet<IndexResult<? extends Addon>>();
            Indexer idx = new Indexer(this.contentRepo, this.content, (Indexer.IndexerEvents)new IndexedCollector(job, paths, indexResults));
            try {
                idx.index(false, true, 1, job.forcedType, null, paths);
                if (!indexResults.isEmpty()) {
                    job.log(Submissions.JobState.SUBMITTING, "Submitting content and opening pull request");
                    try {
                        this.addAndPush(job, indexResults);
                        this.createPullRequest(job, branchName, indexResults);
                        job.log(Submissions.JobState.SUBMITTED, "Submission completed");
                    }
                    catch (Exception e) {
                        job.log(Submissions.JobState.SUBMIT_FAILED, String.format("Submission failed: %s", e.getMessage()), e);
                    }
                }
            }
            catch (Exception e) {
                job.log(Submissions.JobState.INDEX_FAILED, String.format("Content indexing failed: %s", e.getMessage()), e);
                logger.warn("Content index failed", (Throwable)e);
            }
            HashSet<IndexResult<? extends Addon>> hashSet = indexResults;
            return hashSet;
        }
        finally {
            this.checkout(GIT_DEFUALT_BRANCH, false);
        }
    }

    private void checkout(String branchName, boolean createBranch) throws GitAPIException {
        this.gitRepo.checkout().setName(branchName).setCreateBranch(createBranch).call();
    }

    private void addAndPush(Submissions.Job job, Set<IndexResult<? extends Addon>> indexResults) throws GitAPIException {
        Status untrackedStatus = this.gitRepo.status().call();
        if (untrackedStatus.getUntracked().isEmpty()) {
            throw new IllegalStateException("There are no new files to add");
        }
        logger.info("[{}] Adding untracked files: {}", (Object)job.id, (Object)String.join((CharSequence)", ", untrackedStatus.getUntracked()));
        this.gitRepo.add().addFilepattern("content").call();
        job.log("Commit changes to content data");
        this.gitRepo.commit().setCommitter(this.gitAuthor).setAuthor(this.gitAuthor).setMessage(String.format("Add content %s", indexResults.stream().map(i -> String.format("[%s %s] %s", Games.byName((String)i.content.game).shortName, i.content.contentType, i.content.name)).collect(Collectors.joining(", ")))).call();
        job.log("Push content data changes ...");
        ((PushCommand)this.gitRepo.push().setRemote(this.repoUrl).setCredentialsProvider(this.gitCredentials)).call();
        job.log("Content data changes pushed");
    }

    private void createPullRequest(Submissions.Job job, String branchName, Set<IndexResult<? extends Addon>> indexResults) throws IOException {
        job.log("Creating Pull Request for content data change");
        long start = job.log.getFirst().time;
        GHPullRequest pullRequest = this.repository.createPullRequest(branchName, branchName, GIT_DEFUALT_BRANCH, String.format("Add content: %n%s%n%n---%nJob log:%n```%n%s%n```%n%n---%nSubmission log: %s/#%s", indexResults.stream().map(i -> String.format(" - [%s %s] '%s' by '%s' [%s, %s]", Games.byName((String)i.content.game).shortName, i.content.contentType, i.content.name, i.content.author, i.content.hash.substring(0, 8), i.content.isVariation() ? "Variation" : "Original")).collect(Collectors.joining(String.format("%n - ", new Object[0]))), job.log().stream().map(l -> String.format("[%s %.2fs] %s", Character.valueOf(l.type.toString().charAt(0)), Float.valueOf((float)(l.time - start) / 1000.0f), l.message)).collect(Collectors.joining("\n")), SUBMISSION_URL, job.id));
        job.log(String.format("Created Pull Request at %s", pullRequest.getHtmlUrl()));
    }

    private record IndexedCollector(Submissions.Job job, Path[] paths, Set<IndexResult<? extends Addon>> indexResults) implements Indexer.IndexerEvents
    {
        public void starting(int foundFiles) {
            this.job.log(Submissions.JobState.INDEXING, "Begin indexing content");
            logger.info("[{}] Start indexing paths {}", (Object)this.job.id, (Object)Arrays.toString(this.paths));
        }

        public void progress(int indexed, int total, Path currentFile) {
            logger.info("[{}] Indexed {} of {}", new Object[]{this.job.id, indexed, total});
        }

        public void indexed(Submission submission, Optional<IndexResult<? extends Addon>> indexed, IndexLog log) {
            indexed.ifPresentOrElse(i -> {
                this.job.log(String.format("Indexed %s: %s by %s", i.content.contentType, i.content.name, i.content.author));
                this.indexResults.add((IndexResult<? extends Addon>)i);
            }, () -> {
                this.job.log(String.format("Failed to index content in file %s", Util.fileName((Path)submission.filePath)));
                logger.warn(log.log.stream().map(l -> l.message).collect(Collectors.joining("; ")));
            });
        }

        public void completed(int indexedFiles, int errorCount) {
            this.job.log("Indexing complete");
            logger.info("[{}] Completed indexing {} files with {} errors", new Object[]{this.job.id, indexedFiles, errorCount});
        }
    }
}

