/*
 * Decompiled with CFR 0.152.
 */
package org.unrealarchive.submitter;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.Closeable;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.unrealarchive.submitter.ClamScan;
import org.unrealarchive.submitter.ContentRepository;
import org.unrealarchive.submitter.Submissions;

public class SubmissionProcessor
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(SubmissionProcessor.class);
    private static final PendingSubmission[] PENDING_ARRAY = new PendingSubmission[0];
    private static final Duration POLL_WAIT = Duration.ofSeconds(5L);
    private static final Duration SWEEP_RATE = Duration.ofSeconds(120L);
    private static final Duration SWEEP_AGE = Duration.ofHours(36L);
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private final BlockingDeque<PendingSubmission> pending;
    private final ContentRepository repo;
    private final ClamScan clamScan;
    private final Path jobsPath;
    private final Map<String, Submissions.Job> jobs;
    private volatile boolean stopped;

    public SubmissionProcessor(ContentRepository repo, ClamScan clamScan, int queueSize, final ScheduledExecutorService executor, Path jobsPath) {
        this.repo = repo;
        this.clamScan = clamScan;
        this.jobs = new HashMap<String, Submissions.Job>();
        this.pending = new LinkedBlockingDeque<PendingSubmission>(queueSize);
        this.jobsPath = jobsPath;
        this.stopped = false;
        Runnable processor = new Runnable(){

            @Override
            public void run() {
                block9: {
                    if (SubmissionProcessor.this.stopped) {
                        return;
                    }
                    try {
                        PendingSubmission sub = SubmissionProcessor.this.pending.pollFirst(POLL_WAIT.toMillis(), TimeUnit.MILLISECONDS);
                        if (sub == null) break block9;
                        try {
                            sub.job.log("Picked up for processing");
                            SubmissionProcessor.this.process(sub);
                        }
                        catch (Exception e) {
                            sub.job.log(Submissions.JobState.FAILED, String.format("Failed to process submission: %s", e.getMessage()), e);
                            logger.warn("Submission processing failure", (Throwable)e);
                        }
                        finally {
                            SubmissionProcessor.this.writeJob(sub);
                        }
                    }
                    catch (InterruptedException e) {
                        logger.warn("Submission queue processing failure", (Throwable)e);
                    }
                }
                if (!SubmissionProcessor.this.stopped) {
                    executor.submit(this);
                }
            }
        };
        Runnable cleaner = () -> {
            if (this.stopped) {
                return;
            }
            this.jobs.entrySet().removeIf(e -> {
                Submissions.Job job = (Submissions.Job)e.getValue();
                Submissions.LogEntry last = job.log.getLast();
                return last.time < System.currentTimeMillis() - SWEEP_AGE.toMillis();
            });
        };
        executor.submit(processor);
        executor.scheduleAtFixedRate(cleaner, SWEEP_RATE.toMillis(), SWEEP_RATE.toMillis(), TimeUnit.MILLISECONDS);
        logger.info("Submission processor started");
    }

    public PendingSubmission[] pending() {
        return this.pending.toArray(PENDING_ARRAY);
    }

    public boolean trackJob(Submissions.Job job) {
        return this.jobs.put(job.id, job) == null;
    }

    public Collection<Submissions.Job> jobs() {
        return Collections.unmodifiableCollection(this.jobs.values());
    }

    public Submissions.Job job(String jobId) {
        return this.jobs.get(jobId);
    }

    public boolean add(PendingSubmission submission) {
        return this.pending.offerLast(submission);
    }

    @Override
    public void close() {
        this.stopped = true;
    }

    private void writeJob(PendingSubmission submission) {
        try {
            String fName = String.format("%d-%s.json", submission.submitTime, submission.job.id);
            Files.write(this.jobsPath.resolve(fName), MAPPER.writeValueAsBytes((Object)submission), new OpenOption[0]);
        }
        catch (Exception e) {
            logger.warn("Failed to write job file", (Throwable)e);
        }
    }

    private void process(PendingSubmission submission) {
        switch (submission.job.state) {
            case CREATED: {
                if (this.virusScan(submission)) {
                    this.add(submission);
                    break;
                }
                this.fileCleanup(submission);
                break;
            }
            case VIRUS_FREE: {
                if (this.scan(submission)) {
                    this.add(submission);
                    break;
                }
                this.fileCleanup(submission);
                break;
            }
            case SCANNED: {
                this.index(submission);
                this.fileCleanup(submission);
                break;
            }
            default: {
                submission.job.log("Invalid processing state " + String.valueOf((Object)submission.job.state), Submissions.LogType.ERROR);
            }
        }
    }

    private boolean virusScan(PendingSubmission submission) {
        return this.clamScan.scan(submission.job, submission.files) == ClamScan.ClamResult.OK;
    }

    private boolean scan(PendingSubmission submission) {
        if (submission.job.forcedType != null) {
            submission.job.log(Submissions.JobState.SCANNED, "Content scan skipped, forcing type to " + submission.job.forcedType.name());
            return true;
        }
        try {
            this.repo.scan(submission.job, submission.files);
            return submission.job.state == Submissions.JobState.SCANNED;
        }
        catch (IOException e) {
            submission.job.log(Submissions.JobState.SCAN_FAILED, "Scanning failed", e);
            logger.warn("Submission scanning failure", (Throwable)e);
            return false;
        }
    }

    private void index(PendingSubmission submission) {
        this.repo.lock();
        try {
            if (!this.repo.submit(submission.job, submission.files).isEmpty()) {
                submission.job.log(Submissions.JobState.COMPLETED, "Complete!", Submissions.LogType.GOOD);
            } else {
                submission.job.log(Submissions.JobState.FAILED, "No content was added", Submissions.LogType.ERROR);
                logger.warn("Content index returned an empty result");
            }
        }
        catch (Exception e) {
            submission.job.log(Submissions.JobState.FAILED, String.format("Failed to index or submit content: %s", e.getMessage()), e);
            logger.warn("Submission indexing failure", (Throwable)e);
        }
        finally {
            this.repo.unlock();
        }
    }

    private void fileCleanup(PendingSubmission submission) {
        for (Path file : submission.files) {
            try {
                Files.deleteIfExists(file);
            }
            catch (IOException e) {
                logger.warn("Failed to delete file {} for job {}", (Object)file, (Object)submission.job.id);
            }
        }
    }

    static {
        MAPPER.configure(SerializationFeature.INDENT_OUTPUT, true);
    }

    public record PendingSubmission(Submissions.Job job, long submitTime, String name, Path[] files) {
    }
}

